
********************************************************************************
*                                                                              *
* 1 by 4 between subject designs: Examples 1 to 4                              *
*                                                                              *
********************************************************************************

clear
set more off

use between1by4data

browse

// Contrast analysis using the L and M matrix is implemented in Stata in the
// manovatest postestimation command of the MANOVA procedure. In contrast to
// SPSS, Stata does not automaticly provide the C-estimate. The C-estimate, and
// also ETA^2 altering can be calculated manually. If you want to have Stata
// perform the calculations, then it is easiest to do so using the Mata 
// language. The required steps and syntax are explained below. 


**********************
* Conventional ANOVA *
**********************

// First run the traditional ANOVA to obtain SSmodel:

anova grade location

// SSmodel is stored as e(mss) in Stata's memory. To display type:

display e(mss) 

// Since SSmodel is needed to calculate ETA^2 alerting later on, let us for
// now store e(mss) as a scalar named SSmodel in Mata's memory:

mata
SSmodel = st_numscalar("e(mss)") // st_numscalar reads scalars produced in Stata 
							     // into Mata.
end 

**********************************
* Example 1: A first-row effect? *
**********************************

// As indicated above, contrast analysis using the L and M matrix is implemented
// in Stata in the manovatest postestimation command of MANOVA. Therefore, we
// first run a MANOVA with a single dependent variable grade, and Location as
// factor:

manova grade = location

// The output of this MANOVA is of little interest and can be disregarded. 

// Before we can run the manovatest postestimation command, we need to specify
// the row vector L with contrast weights. In contrast to SPSS, this vector L
// also contains the weight of the intercept or grand mean. By using the  
// showorder option of the manovatest postestimation function, one can determine 
// in what order the contrast weights, including that of the intercept, need to 
// be placed in the L matrix:

manovatest, showorder

// It shows that the weight of the intercept (_cons) is placed last

// Define the required L matrix consisting of the contrast weights of the four
// groups in consecutive order and with the weight of the intercept last:

matrix L = (1, -1/3, -1/3, -1/3, 0)

// Test the contrast using the postestimation function manovatest by using the 
// the vector L as input to the test option:

manovatest, test(L)

// As typical in MANOVA output there are four significance tests (including  
// Wilks' lambda and Pillai's trace) and their F approximations. With contrast 
// they typically give the same F-statistic and p-value.

// The SScontrast is not shown in the manovatest output, but is stored in a 
// matrix rH in Stata's memory. It contains as ingle value. To have it 
// displayed, type:

matrix list r(H) 

// Now store it in Mata's memory as a matrix labeled SScontrast. This way it can 
// be used for the calculation of the ETA^2 alterting later on:

mata
SScontrast = st_matrix("r(H)")
end

// Stata does not provide the C-estimate when using the manovatest command. To
// have Stata to calculate it, first obtain the observed group or cell means:

mean grade, over(location)

// The means are stored in a vector e(b) in Stata's memory. 

// Next we can obtain the C-estimate by calculating the weighted sum of cell 
// means using the contrast coefficients in vector L as weights:

mata
Means = st_matrix("e(b)") // Make the vector with means available to Mata
L = st_matrix("L")        // Make the vector with contrast weigths L available 
                          // to Mata 
L = L[1,1::(cols(L) - 1)] // Select the weights of the cell means from L 
						  // That is, select all except the last collumn in L
Cest = rowsum(L :* Means) // Calculate the C-estimate; note that :* means 
						  // element-wise multiplication and that rowsum will 
						  // sum all weighted cell means.
Cest					  // display the C-estimate
end

// Next let us use Mata to calculate and display the ETA^2 alerting:

mata
Eta2alerting = SScontrast/SSmodel
Eta2alerting
end

***********************************************
* Example 2: Rows 2 and 3 outperform 1 and 4? *
***********************************************

// Testing the theorectical prediction described in Example 2 requires the same
// syntax as used in Example 1, except of course for a different specification 
// of the vector L with contrast weights. Note also that SSmodel and the 
// observed cell means are already stored in Mata from the analyses of 
// Example 1:

matrix L = (-1/2, 1/2, 1/2, -1/2, 0)

manova grade = location
manovatest, test(L)

mata
// Obtain SScontrast:
SScontrast = st_matrix("r(H)")
SScontrast
// Calculate C-estimate:
L = st_matrix("L") 
Cest = rowsum(L = L[1,1::(cols(L) - 1)] :* Means) 	  
Cest	
// Calculate ETA^2 alerting:
Eta2alerting = SScontrast/SSmodel
Eta2alerting				
end

***************************************
* Example 3: A negative linear trend? *
***************************************

// For Example 3, again, nothing has changed except the contrast weights 
// specified in the vector L:

matrix L =  (3, 1, -1, -3, 0)
manova grade = location
manovatest, test(L)

mata
SScontrast = st_matrix("r(H)")
SScontrast
L = st_matrix("L") 
Cest = rowsum(L = L[1,1::(cols(L) - 1)] :* Means) 	  
Cest	
Eta2alerting = SScontrast/SSmodel
Eta2alerting				
end

**********************************
* Example 4: Combining contrasts *
**********************************

// To test against the emperical data a prediction based on two theories, the 
// set of contrast weights L becomes a matrix with in each row the contrast 
// weights of a single theorectical prediction. In contrast to SPSS, Stata will
// only perform the test of the combined prediction, and will not provide tests 
// of each individual prediction. If needed, then these can of course be
// obtained by running separate analyses for each (see Examples 2 and 3). 

// Let us first specify two vectors L1 and L2 with contrast weights; one for 
// each of the two predictions:

matrix L1 = (-1/2, 1/2, 1/2, -1/2, 0)
matrix L2 = (3, 1, -1, -3, 0)

// Next, we combine vector L1 and L2 into a single matrix L:

matrix L = (L1 \ L2)

// Next, we run the manovatest postestimation command of the MANOVA procedure, 
// using the matrix L as input to the test option:

manova grade = location
manovatest, test(L)

// Next, obtain SScontrast and ETA^2 altering of the combined prediction using
// the same Mata syntax as used in Examples 1 to 3:

mata
SScontrast = st_matrix("r(H)")
SScontrast
Eta2alerting = SScontrast/SSmodel
Eta2alerting
end


